<?php

namespace App\Utils;

use App\Business;
use App\Notifications\CustomerNotification;
use App\Notifications\RecurringExpenseNotification;
use App\Notifications\RecurringInvoiceNotification;
use App\Notifications\SupplierNotification;
use App\NotificationTemplate;
use App\Restaurant\Booking;
use App\System;
use Config;
use Notification;
use App\Contact;
use Illuminate\Support\Facades\Cache;

class NotificationUtil extends Util
{
    /**
     * Automatically send notification to customer/supplier if enabled in the template setting
     *
     * @param  int  $business_id
     * @param  string  $notification_type
     * @param  obj  $transaction
     * @param  obj  $contact
     * @return void|string  WhatsApp link if applicable
     */
    public function autoSendNotification($business_id, $notification_type, $transaction, $contact)
    {
        if (!config('constants.enable_auto_notifications')) {
            return '';
        }

        $notification_template = Cache::remember(
            "notif:tpl:$business_id:$notification_type",
            600,
            fn () => NotificationTemplate::where('business_id', $business_id)
                ->where('template_for', $notification_type)
                ->first()
        );

        $business = Cache::remember(
            "notif:biz:$business_id",
            600,
            fn () => Business::select('id', 'email_settings', 'sms_settings')->findOrFail($business_id)
        );
        $data['email_settings'] = $business->email_settings;
        $data['sms_settings']   = $business->sms_settings;

        $whatsapp_link = '';

        if (!empty($notification_template)) {
            if (!empty($notification_template->auto_send)
                || !empty($notification_template->auto_send_sms)
                || !empty($notification_template->auto_send_wa_notif)) {

                $orig_data = [
                    'email_body'    => $notification_template->email_body,
                    'sms_body'      => $notification_template->sms_body,
                    'subject'       => $notification_template->subject,
                    'whatsapp_text' => $notification_template->whatsapp_text,
                ];

                $tag_replaced_data = $this->replaceTags($business_id, $orig_data, $transaction);

                $data['email_body']    = $tag_replaced_data['email_body'];
                $data['sms_body']      = $tag_replaced_data['sms_body'];
                $data['whatsapp_text'] = $tag_replaced_data['whatsapp_text'];

                // Auto send email
                if (!empty($notification_template->auto_send) && !empty($contact->email)) {
                    $data['subject']  = $tag_replaced_data['subject'];
                    $data['to_email'] = $contact->email;

                    $customer_notifications = NotificationTemplate::customerNotifications();
                    $supplier_notifications = NotificationTemplate::supplierNotifications();

                    try {
                        if (array_key_exists($notification_type, $customer_notifications)) {
                            Notification::route('mail', $data['to_email'])
                                ->notify(new CustomerNotification($data));
                        } elseif (array_key_exists($notification_type, $supplier_notifications)) {
                            Notification::route('mail', $data['to_email'])
                                ->notify(new SupplierNotification($data));
                        }
                        $this->activityLog($transaction, 'email_notification_sent', null, [], false, $business_id);
                    } catch (\Exception $e) {
                        \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());
                    }
                }

                // Auto send sms
                if (!empty($notification_template->auto_send_sms)) {
                    $data['mobile_number'] = $contact->mobile;
                    if (!empty($contact->mobile)) {
                        try {
                            $this->sendSms($data);
                            $this->activityLog($transaction, 'sms_notification_sent', null, [], false, $business_id);
                        } catch (\Exception $e) {
                            \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());
                        }
                    }
                }

                // WhatsApp link
                if (!empty($notification_template->auto_send_wa_notif)) {
                    $data['mobile_number'] = $contact->mobile;
                    if (!empty($contact->mobile)) {
                        $whatsapp_link = $this->getWhatsappNotificationLink($data);
                    }
                }
            }
        }

        return $whatsapp_link;
    }

    /**
     * Replaces tags from notification body with original value (Restaurant Booking)
     *
     * @param  int    $business_id
     * @param  array  $data
     * @param  int    $booking_id
     * @return array
     */
    public function replaceBookingTags($business_id, $data, $booking_id)
    {
        $business = Business::findOrFail($business_id);
        $booking = Booking::where('business_id', $business_id)
            ->with(['customer', 'table', 'correspondent', 'waiter', 'location', 'business'])
            ->findOrFail($booking_id);

        foreach ($data as $key => $value) {
            // Replace contact name
            if (strpos($value, '{contact_name}') !== false) {
                $contact_name = $booking->customer->name;
                $data[$key] = str_replace('{contact_name}', $contact_name, $data[$key]);
            }

            if (strpos($value, '{contact_custom_field_1}') !== false) {
                $val = $booking->customer->custom_field1 ?? '';
                $data[$key] = str_replace('{contact_custom_field_1}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_2}') !== false) {
                $val = $booking->customer->custom_field2 ?? '';
                $data[$key] = str_replace('{contact_custom_field_2}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_3}') !== false) {
                $val = $booking->customer->custom_field3 ?? '';
                $data[$key] = str_replace('{contact_custom_field_3}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_4}') !== false) {
                $val = $booking->customer->custom_field4 ?? '';
                $data[$key] = str_replace('{contact_custom_field_4}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_5}') !== false) {
                $val = $booking->customer->custom_field5 ?? '';
                $data[$key] = str_replace('{contact_custom_field_5}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_6}') !== false) {
                $val = $booking->customer->custom_field6 ?? '';
                $data[$key] = str_replace('{contact_custom_field_6}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_7}') !== false) {
                $val = $booking->customer->custom_field7 ?? '';
                $data[$key] = str_replace('{contact_custom_field_7}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_8}') !== false) {
                $val = $booking->customer->custom_field8 ?? '';
                $data[$key] = str_replace('{contact_custom_field_8}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_9}') !== false) {
                $val = $booking->customer->custom_field9 ?? '';
                $data[$key] = str_replace('{contact_custom_field_9}', $val, $data[$key]);
            }
            if (strpos($value, '{contact_custom_field_10}') !== false) {
                $val = $booking->customer->custom_field10 ?? '';
                $data[$key] = str_replace('{contact_custom_field_10}', $val, $data[$key]);
            }

            // Table
            if (strpos($value, '{table}') !== false) {
                $table = !empty($booking->table->name) ? $booking->table->name : '';
                $data[$key] = str_replace('{table}', $table, $data[$key]);
            }

            // Times
            if (strpos($value, '{start_time}') !== false) {
                $start_time = $this->format_date($booking->booking_start, true);
                $data[$key] = str_replace('{start_time}', $start_time, $data[$key]);
            }
            if (strpos($value, '{end_time}') !== false) {
                $end_time = $this->format_date($booking->booking_end, true);
                $data[$key] = str_replace('{end_time}', $end_time, $data[$key]);
            }

            // Location
            if (strpos($value, '{location}') !== false) {
                $data[$key] = str_replace('{location}', $booking->location->name, $data[$key]);
            }
            if (strpos($value, '{location_name}') !== false) {
                $data[$key] = str_replace('{location_name}', $booking->location->name, $data[$key]);
            }
            if (strpos($value, '{location_address}') !== false) {
                $data[$key] = str_replace('{location_address}', $booking->location->location_address, $data[$key]);
            }
            if (strpos($value, '{location_email}') !== false) {
                $data[$key] = str_replace('{location_email}', $booking->location->email, $data[$key]);
            }
            if (strpos($value, '{location_phone}') !== false) {
                $data[$key] = str_replace('{location_phone}', $booking->location->mobile, $data[$key]);
            }
            if (strpos($value, '{location_custom_field_1}') !== false) {
                $data[$key] = str_replace('{location_custom_field_1}', $booking->location->custom_field1, $data[$key]);
            }
            if (strpos($value, '{location_custom_field_2}') !== false) {
                $data[$key] = str_replace('{location_custom_field_2}', $booking->location->custom_field2, $data[$key]);
            }
            if (strpos($value, '{location_custom_field_3}') !== false) {
                $data[$key] = str_replace('{location_custom_field_3}', $booking->location->custom_field3, $data[$key]);
            }
            if (strpos($value, '{location_custom_field_4}') !== false) {
                $data[$key] = str_replace('{location_custom_field_4}', $booking->location->custom_field4, $data[$key]);
            }

            // Staff
            if (strpos($value, '{service_staff}') !== false) {
                $name = !empty($booking->waiter) ? $booking->waiter->user_full_name : '';
                $data[$key] = str_replace('{service_staff}', $name, $data[$key]);
            }
            if (strpos($value, '{correspondent}') !== false) {
                $name = !empty($booking->correspondent) ? $booking->correspondent->user_full_name : '';
                $data[$key] = str_replace('{correspondent}', $name, $data[$key]);
            }

            // Business
            if (strpos($value, '{business_name}') !== false) {
                $data[$key] = str_replace('{business_name}', $business->name, $data[$key]);
            }
            if (strpos($value, '{business_logo}') !== false) {
                $logo_name = $business->logo;
                // Build absolute HTTPS URL when app.url is https to avoid blocked images in email clients
                if (!empty($logo_name)) {
                    $appUrl = (string) config('app.url');
                    $logo_url = asset('uploads/business_logos/'.$logo_name);
                    if (is_string($appUrl) && stripos($appUrl, 'https://') === 0) {
                        $logo_url = secure_asset('uploads/business_logos/'.$logo_name);
                    }
                    $business_logo = '<img src="'.$logo_url.'" alt="Business Logo" style="max-width: 320px; height: auto;" >';
                } else {
                    $business_logo = '';
                }
                $data[$key] = str_replace('{business_logo}', $business_logo, $data[$key]);
            }
        }

        return $data;
    }

    public function recurringInvoiceNotification($user, $invoice)
    {
        $user->notify(new RecurringInvoiceNotification($invoice));
    }

    public function recurringExpenseNotification($user, $expense)
    {
        $user->notify(new RecurringExpenseNotification($expense));
    }

    /**
     * Apply email settings (from business/session or superadmin) to runtime config
     * Uses modern Laravel mail keys: mail.default, mail.mailers.smtp.*, mail.from.*
     */
    public function configureEmail($notificationInfo = [], $check_superadmin = true)
    {
        $email_settings = !empty($notificationInfo['email_settings']) ? $notificationInfo['email_settings'] : [];

        if (empty($email_settings) && session()->has('business')) {
            $email_settings = request()->session()->get('business.email_settings');
        }

        $is_superadmin_settings_allowed = System::getProperty('allow_email_settings_to_businesses');

        // Prefer superadmin email settings if allowed & requested
        if (!empty($is_superadmin_settings_allowed) && !empty($email_settings['use_superadmin_settings']) && $check_superadmin) {
            $email_settings['mail_driver']       = config('mail.default', 'smtp');
            $email_settings['mail_host']         = config('mail.mailers.smtp.host');
            $email_settings['mail_port']         = config('mail.mailers.smtp.port');
            $email_settings['mail_username']     = config('mail.mailers.smtp.username');
            $email_settings['mail_password']     = config('mail.mailers.smtp.password');
            $email_settings['mail_encryption']   = config('mail.mailers.smtp.encryption');
            $email_settings['mail_from_address'] = config('mail.from.address');
            $email_settings['mail_from_name']    = config('mail.from.name', config('app.name'));
        }

        // Normalize port/encryption combo for common providers (e.g., Gmail)
        $port = (int)($email_settings['mail_port'] ?? 587);
        $enc  = strtolower((string)($email_settings['mail_encryption'] ?? ''));

        if ($port === 465) {
            $enc = 'ssl';
        } elseif ($port === 587) {
            // if 587 and not explicitly ssl, default to tls
            $enc = ($enc === 'ssl') ? 'ssl' : 'tls';
        } elseif ($enc === '') {
            $enc = null; // allow unencrypted if explicitly blank
        }

        // Apply using NEW keys
        Config::set('mail.default', $email_settings['mail_driver'] ?? 'smtp');
        Config::set('mail.mailers.smtp.host', $email_settings['mail_host'] ?? 'smtp.gmail.com');
        Config::set('mail.mailers.smtp.port', $port);
        Config::set('mail.mailers.smtp.encryption', $enc ?: null);
        Config::set('mail.mailers.smtp.username', $email_settings['mail_username'] ?? null);
        Config::set('mail.mailers.smtp.password', $email_settings['mail_password'] ?? null);

        // From header (fallback to username if address missing)
        $fromAddress = $email_settings['mail_from_address'] ?? $email_settings['mail_username'] ?? null;
        $fromName    = $email_settings['mail_from_name'] ?? config('app.name');

        Config::set('mail.from.address', $fromAddress);
        Config::set('mail.from.name', $fromName);
    }

    public function replaceHmsBookingTags($data, $transaction, $adults, $childrens, $customer)
    {
        $business = Business::findOrFail($transaction->business_id);

        foreach ($data as $key => $value) {
            // Customer
            if (strpos($value, '{customer_name}') !== false) {
                $data[$key] = str_replace('{customer_name}', $customer->name, $data[$key]);
            }

            // Business name (deduped)
            if (strpos($value, '{business_name}') !== false) {
                $data[$key] = str_replace('{business_name}', $business->name, $data[$key]);
            }

            // Business logo
            if (strpos($value, '{business_logo}') !== false) {
                $logo_name = $business->logo;
                if (!empty($logo_name)) {
                    $appUrl = (string) config('app.url');
                    $logo_url = asset('uploads/business_logos/'.$logo_name);
                    if (is_string($appUrl) && stripos($appUrl, 'https://') === 0) {
                        $logo_url = secure_asset('uploads/business_logos/'.$logo_name);
                    }
                    $business_logo = '<img src="'.$logo_url.'" alt="Business Logo" style="max-width: 320px; height: auto;" >';
                } else {
                    $business_logo = '';
                }
                $data[$key] = str_replace('{business_logo}', $business_logo, $data[$key]);
            }

            // Booking details
            if (strpos($value, '{booking_id}') !== false) {
                $data[$key] = str_replace('{booking_id}', $transaction->ref_no, $data[$key]);
            }
            if (strpos($value, '{booking_status}') !== false) {
                $data[$key] = str_replace('{booking_status}', $transaction->status, $data[$key]);
            }
            if (strpos($value, '{arrival_date}') !== false) {
                $start_date = $this->format_date($transaction->hms_booking_arrival_date_time, true);
                $data[$key] = str_replace('{arrival_date}', $start_date, $data[$key]);
            }
            if (strpos($value, '{departure_date}') !== false) {
                $end_time = $this->format_date($transaction->hms_booking_departure_date_time, true);
                $data[$key] = str_replace('{departure_date}', $end_time, $data[$key]);
            }

            // Counts
            if (strpos($value, '{adults}') !== false) {
                $data[$key] = str_replace('{adults}', $adults, $data[$key]);
            }
            if (strpos($value, '{childrens}') !== false) {
                $data[$key] = str_replace('{childrens}', $childrens, $data[$key]);
            }
        }

        return $data;
    }
}
